package gov.va.genisis2.dao.impl;

import gov.va.genisis2.dao.ICommentHistoryDao;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.CommentHistory;

import java.util.List;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class CommentHistoryDao.
 * 
 * The CommentHistory data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, CommentHistory DAO
 * provide some specific data operations without exposing details of the
 * database.
 */
@Repository
@Transactional(value = "transactionManager")
public class CommentHistoryDao implements ICommentHistoryDao {
	
	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(CommentHistoryDao.class);
	
	@Autowired
	private SessionFactory sessionFactory;

	/**
	 * This method is used to create commentHistory.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns created comment id.
	 */
	@Override
	public int createCommentHistory(CommentHistory entity) throws GenisisDAOException {
		// NOT BEING USED
		if (entity == null) {
			return 0;
		}
		Integer commentHistoryId = this.maxRowValueCommentHistory();
		if (commentHistoryId == 0) {
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("Exception occurred while getting Max value on Comment History table.Unable to create Comment History");
			}
			return 0;
		} else
			entity.setCommentId(commentHistoryId.intValue());
		try {
			Session session = sessionFactory.getCurrentSession();
			commentHistoryId = (Integer) session.save(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on createCommentHistory.", ex);
			throw new GenisisDAOException("Exception occurred on createCommentHistory.", ex);
		}
		return commentHistoryId.intValue();
	}

	/**
	 * This method is used to updated commentHistory.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns updated comment id.
	 */
	@Override
	public int updateCommentHistory(CommentHistory entity) throws GenisisDAOException {
		// NOT BEING USED
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("updateCommentHistory ");
		}
		
		try {
			Session session = sessionFactory.getCurrentSession();
			session.update(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on updateCommentHistory.", ex);
			throw new GenisisDAOException("Exception occurred on updateCommentHistory.", ex);
		}
		CommentHistory commentHistory = this.getCommentHistoryByCommentId(entity.getCommentId());
		return (null != commentHistory) ? commentHistory.getCommentId() : null;
	}

	/**
	 * This method is used to get list of commentHistory by UID.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoriesByUID(String uid) throws GenisisDAOException {
		// NOT BEING USED
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getCommentHistoriesByUID ");
		}

		List<CommentHistory> listCommentHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
			Root<CommentHistory> root = query.from(CommentHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get("createdBy"), uid);
			Predicate conditions = criteriaBuilder.and(condition1);
			query.where(conditions);
			listCommentHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getCommentHistoriesByUID.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getCommentHistoriesByUID.", ex);
		}
		return listCommentHistory;
	}

	/**
	 * This method is used to getCommentHistories.
	 * 
	 * @return this returns list of commentHistories.
	 */
	/*@Override
	public List<CommentHistory> getCommentHistories() throws GenisisDAOException {
		// NOT BEING USED
		LOGGER.info("getCommentHistories ");
		List<CommentHistory> listCommentHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
			listCommentHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getCommentHistories.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getCommentHistories.", ex);
		}
		return listCommentHistory;
	}*/

	/**
	 * This method is used to submit or modify the table.
	 * 
	 * @param entity
	 *            The entity.
	 * @return int This returns modified commentHistory id.
	 */
	@Override
	public int submitOrModify(CommentHistory entity) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("submitOrModify (CommentHistory) ");
		}

		Integer commentHistoryId;
		if (entity.getCommentId() == 0) {
			commentHistoryId = this.maxRowValueCommentHistory();
			entity.setCommentId(commentHistoryId);
		}
		try {
			Session session = sessionFactory.getCurrentSession();
			session.saveOrUpdate(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify CommentHistory.", ex);
			throw new GenisisDAOException("Exception occurred on submitOrModify CommentHistory.", ex);
		}
		return entity.getCommentId();
	}

	/**
	 * This method is used to get list of commentHistories by id.
	 * 
	 * @param requestId
	 *            The requestId.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoryById(int requestId) throws GenisisDAOException {
		// NOT BEING USED
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getCommentHistoryById ");
		}
		
		List<CommentHistory> listCommentHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
			Root<CommentHistory> root = query.from(CommentHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get("request").get("id"), requestId);
			Predicate conditions = criteriaBuilder.and(condition1);
			query.where(conditions);
			listCommentHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getCommentHistoryById.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getCommentHistoryById.", ex);
		}
		return listCommentHistory;
	}

	/**
	 * This method is used to get commentHistory by comment id.
	 * 
	 * @param commentId
	 *            The commentId.
	 * @return CommentHistory This returns commentHistory.
	 */
	@Override
	public CommentHistory getCommentHistoryByCommentId(int commentId) throws GenisisDAOException {
		// NOT BEING USED
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getCommentHistoryByCommentId ");
		}

		CommentHistory commentHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
			Root<CommentHistory> root = query.from(CommentHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get("commentId"), commentId);
			Predicate conditions = criteriaBuilder.and(condition1);
			query.where(conditions);
			commentHistory = session.createQuery(query).getSingleResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getCommentHistoryByCommentId.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getCommentHistoryByCommentId.", ex);
		}
		return commentHistory;
	}

	/**
	 * This method is used to get list of commentHistories by request id and
	 * status id.
	 * 
	 * @param requestId
	 *            The requestId.
	 * @param taskId
	 *            The taskId.
	 * @return this returns list of commentHistories.
	 */
	@Override
	public List<CommentHistory> getCommentHistoryByRequestIdStatusId(int requestId, String taskId) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getCommentHistoryByRequestIdStatusId ");
		}

		List<CommentHistory> listCommentHistory = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<CommentHistory> query = criteriaBuilder.createQuery(CommentHistory.class);
			Root<CommentHistory> root = query.from(CommentHistory.class);
			Predicate condition1 = criteriaBuilder.equal(root.get("request").get("id"), requestId);
			Predicate condition2 = criteriaBuilder.equal(root.get("taskId"), taskId);
			Predicate conditions = criteriaBuilder.and(condition1, condition2);
			query.where(conditions);
			listCommentHistory = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getCommentHistoryByRequestIdStatusId.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getCommentHistoryByRequestIdStatusId.", ex);
		}
		return listCommentHistory;
	}

	/**
	 * This method is used to get max value record from the table.
	 * 
	 * @return int This returns max value record plus one.
	 */
	private int maxRowValueCommentHistory() {
		// TODO: createCriteria Deprecated
		int maxRowVal = 1;
		Integer result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			@SuppressWarnings("deprecation")
			Criteria criteria = session.createCriteria(CommentHistory.class).setProjection(Projections.max("commentId"));
			result = (Integer) criteria.uniqueResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on maxRowValue().", ex);
		}
		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}
}